using System;
using System.Text;
using System.Collections;

namespace com.swvoyages.reservation {

	///
	/// <summary>La classe <code>ReservationManager</code> implemente 
	/// l'interface <code>ReservationRegistry</code> dont l'objectif 
	/// consiste  grer toutes les rservations en cours dans le processus 
	/// d'excution courant (application).</summary>
	/// <remarks>Cette implmentation tend la classe  
	/// <code>Hashtable</code>.</remarks>
	///
	public class ReservationManager : Hashtable, ReservationRegistry {

		private static ReservationManager uniqueInstance = new ReservationManager();

		///
		/// <summary>Construit une instance de la classe <code>ReservationManager</code>.
		/// </summary>
		/// 
		private ReservationManager() : base() {
		}

		///
 		/// <summary>Renvoie le singleton de la classe <code>ReservationManager</code>.
 		/// </summary>
 		/// 
 		/// <returns>le singleton de la classe <code>ReservationManager</code>.
 		/// </returns>
		///
		public static ReservationManager getUniqueInstance() {
			if (uniqueInstance == null) {
				uniqueInstance = new ReservationManager();
				uniqueInstance.initialize();
			}
			return uniqueInstance;
		}

		///
		/// <summary>Purge le registre des rservations.</summary>
		///
		public void clear() {
			Clear();
		}

		///
		/// <summary>Renvoie la table des rservations en cours (clone).
		/// </summary>
		/// 
		/// <returns>la table des rservations en cours<returns>
		///
		public Hashtable getAllReservations() {
			return (Hashtable)Clone();
		}

		///
		/// <summary>Renvoie la rservation correspondant  l'identifiant 
		/// pass en paramtre, si elle existe dans le registre, <b>null</b> 
		/// sinon.</summary>
		///
		/// <param name="reservationId">l'identifiant de la rservation
		/// </param>
		///
		/// <returns>la rservation rfrence par l'identifiant 
		///	reservationId; <b>null</b> sinon</returns>
		///
		public Object getReservation(Object reservationId) {
			return this[(Object)reservationId];
		}

		///
		/// <summary>Initialise le registre des rservations.</summary>
		///
		public void initialize() {
		}

		///
		/// <summary>Renvoie <b>true</b> si le registre rfrence des 
		/// rservations en cours.</summary>
		///
		/// <returns><code>true</code> si le registre contient des
		///	rservations; <code>false</code> sinon<returns>
		///
		public bool isEmpty() {
			if (Count==0) {
				return true;
			}
			return false;
		}

		///
		/// <summary>Renvoie <b>true</b> si la rservation passe en 
		/// paramtre est enregistre.</summary>
		///
		/// <param name="reservation">la rservation  vrifier</param>
		///
		/// <returns><code>true</code> si la rservation est
		///	enregistre; <code>false</code> sinon</returns>
		///
		public bool isReservationRegistered(Object reservation) {
			return ContainsValue(reservation);
		}

		///
		/// <summary>Enregistre une nouvelle rservation.</summary>
		///
		/// <param name="reservationId">l'identifiant de la rservation
		/// </param> 
		/// <param name="reservation">la rservation</param>
		///
		/// <returns>la nouvelle rservation enregistre</returns>
		///
		/// <exception cref="ReservationException">lance si une anomalie 
		/// se produit durant l'enregistrement de la rservation</exception>
		///
		public Object newReservation(Object reservationId, Object reservation) {

			try {
				this[(Object)reservationId]=reservation;
			}
			catch (Exception e) {
				StringBuilder sb = new StringBuilder();
				sb.Append("une ");
				sb.Append(e.GetType().FullName);
				sb.Append(" s'est produite durant l'enregistrement de la rservation");
				if ((e.Message!=null) && (!(e.Message.Equals("")))) {
					sb.Append(" : ");
					sb.Append(e.Message);
				}
				throw new ReservationException(sb.ToString(), e);
			}
			return reservation;
		}

		///
		/// <summary>Libre les rservations du registre.</summary>
		///
		public void release() {
			clear();
		}

		///
		/// <summary>Supprime la rservation identifie du registre.
		/// </summary>
		///
		/// <param name="reservationId">l'identifiant de la rservation
		/// </param>
		///
		public void removeReservation(Object reservationId) {
			Remove(reservationId);
		}

		///
		/// <summary>Renvoie le nombre des rservations en cours.</summary>
		///
		/// <returns>le nombre des rservations en cours</returns>
		///
		public int size() {
			return Count;
		}

		///
		/// <summary>Renvoie une reprsentation textuelle du registre.
		/// </summary> 
		///
		/// <returns>une reprsentation textuelle du registre.</returns>
		///
		public String toString() {
			StringBuilder sb = new StringBuilder();
			sb.Append("\r");
			sb.Append(GetType().FullName);
			sb.Append("@");
			sb.Append(System.Convert.ToString(GetHashCode(), 16));
			return sb.ToString();
		}
	}
}